#***********************************************************************
# JTE_DOM
#     DOM for JTE
#***********************************************************************

$KCODE = 'u'


class JTE_DOM
  XML_ELEMENT_NODE        = 1
  XML_TEXT_NODE           = 3
  XML_CDATA_SECTION_NODE  = 4
  XML_COMMENT_NODE        = 8
  XML_DOCUMENT_NODE       = 9
end


# DOMNode

class JTE_DOMNode
  @@seq = 0 ;
  
  def initialize( nodetype )
    @@seq += 1
    @seq = @@seq
    @nodeType = nodetype
    
    @parentNode = nil
  end
  
  attr_accessor :seq, :nodeType, :ownerDocument, :parentNode
end


# DOMText

class JTE_DOMText < JTE_DOMNode
  def initialize( text = '' )
    super( JTE_DOM::XML_TEXT_NODE )
    @nodeValue = text.clone
  end
  
  def cloneNode( deep = false )
    return JTE_DOMText.new( @nodeValue )
  end
  
  attr_accessor :nodeValue
end


# DOMComment

class JTE_DOMComment < JTE_DOMNode
  def initialize( comment = '' )
    super( JTE_DOM::XML_COMMENT_NODE )
    @nodeValue = comment.clone
  end
  
  def cloneNode( deep = false )
    return JTE_DOMComment.new( @nodeValue )
  end
  
  attr_accessor :nodeValue
end


# DOMCDATASection

class JTE_DOMCDATASection < JTE_DOMNode
  def initialize( cdata = '' )
    super( JTE_DOM::XML_CDATA_SECTION_NODE )
    @nodeValue = cdata.clone
  end
  
  def cloneNode( deep = false )
    return JTE_DOMCDATASection.new( @nodeValue )
  end
  
  attr_accessor :nodeValue
end


# DOMElement

class JTE_DOMElement < JTE_DOMNode
  def initialize( tagName )
    super( JTE_DOM::XML_ELEMENT_NODE )
    @tagName = tagName.clone
    @attributes = JTE_DOMAttributeList.new()
    @childNodes = JTE_DOMNodeList.new()
  end
  
  
  def cloneNode( deep = false )
    clone = JTE_DOMElement.new( @tagName ) ;
    
    for n in 0 ... @attributes.length
      clone.setAttribute( @attributes.item( n ).name, @attributes.item( n ).value )
    end
    
    if deep
      for n in 0 ... @childNodes.length
        clone.appendChild( self.childNodes._item[ n ].cloneNode( true ) )
      end
    end
    
    return clone
  end
  
  
  def hasAttributes()
    return ( @attributes._item.length > 0 )
  end
  
  
  def hasAttribute( name )
    return @attributes._index.key?( name )
  end
  
  
  def setAttribute( name, value )
    if @attributes._index.key?( name )
      @attributes._item[ @attributes._index[ name ] ].value.replace( value.clone )
    else
      @attributes._index[ name ] = @attributes.length
      @attributes._item.push( JTE_DOMAttribute.new( name, value ) )
    end
  end
  
  
  def getAttribute( name )
    if @attributes._index.key?( name )
      return @attributes._item[ @attributes._index[ name ] ].value
    else
      return ''
    end
  end
  
  
  def removeAttribute( name )
    if @attributes._index.key?( name )
      @attributes._item.delete_at( @attributes._index[ name ] )
      @attributes._index.delete( name )
    end
  end
  
  
  def hasChildNodes()
    return ( @childNodes._item.length > 0 )
  end
  
  
  def appendChild( node )
    node.parentNode = self
    
    @childNodes._index[ node.seq ] = @childNodes._item.length
    @childNodes._item.push( node )
    
    return node
  end
  
  
  def insertBefore( node, before )
    unless before.parentNode.seq == @seq
      return nil
    end
    
    unless @childNodes._index.key?( before.seq )
      return nil
    end
    
    node.parentNode = self
    
    @childNodes._item.insert( self.childNodes._index[ before.seq ], node )
    
    @childNodes._index.clear
    
    for n in 0 ... @childNodes._item.length
      @childNodes._index[ @childNodes._item[ n ].seq ] = n
    end
    
    return node
  end
  
  
  def replaceChild( node, replace )
    unless replace.parentNode.seq == @seq
      return nil
    end
    
    unless @childNodes._index.key?( replace.seq )
      return nil
    end
    
    node.parentNode = self
    
    n = @childNodes._index[ replace.seq ]
    @childNodes._index.delete( replace.seq )
    @childNodes._index[ node.seq ] = n
    
    @childNodes._item[ n ] = node
    
    return node
  end
  
  
  def removeChild( node )
    if node.parentNode.seq != @seq
      return nil
    end
    
    unless @childNodes._index.key?( node.seq )
      return nil
    end
    
    @childNodes._item.delete_at( @childNodes._index[ node.seq ] )
    
    @childNodes._index.clear
    
    for n in 0 ... @childNodes._item.length
      @childNodes._index[ @childNodes._item[ n ].seq ] = n
    end
    
    return node
  end
  
  
  def firstChild
    return @childNodes._item.first
  end
  
  
  def lastChild
    return @childNodes._item.last
  end
  
  
  def previousSibling
    if @parentNode == nil
      return nil
    end
    
    if @parentNode.childNodes._index[ @seq ] == 0
      return nil
    end
    
    return @parentNode.childNodes._item[ @parentNode.childNodes._index[ @seq ] - 1 ]
  end
  
  
  def nextSibling
    if @parentNode == nil
      return nil
    end
    
    if @parentNode.childNodes._index[ @seq ] == ( @parentNode.childNodes._item.length - 1 )
      return nil
    end
    
    return @parentNode.childNodes._item[ @parentNode.childNodes._index[ @seq ] + 1 ]
  end
  
  attr_accessor :tagName, :attributes, :childNodes
end


# DOMNodeList

class JTE_DOMNodeList
  def initialize
    @_item = []
    @_index = {}
  end
  
  
  def length
    return @_item.length
  end
  
  
  def item( n )
    return @_item[ n ]
  end
  
  
  attr_accessor :_item, :_index
end


# DOMAttribute

class JTE_DOMAttribute
  def initialize( name, value )
    @name = name.clone
    @value = value.clone
  end
  
  attr_accessor :name, :value
end


# DOMAttributeList

class JTE_DOMAttributeList
  def initialize
    @_item = []
    @_index = {}
  end
  
  
  def length
    return @_item.length
  end
  
  
  def item( n )
    return @_item[ n ]
  end
  
  attr_accessor :_item, :_index
end


# DOMDocumentType

class JTE_DOMDocumentType
  def initialize( docType = '' )
    if docType == ''
      self._parse( '<!DOCTYPE html PUBLIC "-//W3C//DTD XHTML 1.0 Transitional//EN" "http://www.w3.org/TR/xhtml1/DTD/xhtml1-transitional.dtd">' )
    else
      self._parse( docType )
    end
  end
  
  
  def _parse( doctype )
    if doctype.strip =~ /^\<!DOCTYPE[\s\n]+([^\s\n]+)[\s\n]+PUBLIC[\s\n]+("[^"]*")[\s\n]+("[^"]*")[\s\n]*\>$/m
      @internalId = doctype.strip
      @name = $1.downcase
      @publicId = $2
      @systemId = $3
    end
  end
  
  attr_accessor :internalId, :name, :publicId, :systemId
end


# DOMDocument

class JTE_DOMDocument < JTE_DOMNode
  def initialize
    super( JTE_DOM::XML_DOCUMENT_NODE )
    @doctype = JTE_DOMDocumentType.new
    @encoding = 'UTF-8'
    
    @documentElement = nil
  end
  
  
  def appendChild( node )
    node.parentNode = self
    @documentElement = node
    
    return node
  end
  
  
  def createComment( comment = '' )
    node = JTE_DOMComment.new( comment )
    node.ownerDocument = self
    return node
  end
  
  
  def createTextNode( text = '' )
    node = JTE_DOMText.new( text )
    node.ownerDocument = self
    return node
  end
  
  
  def createCDATASenction( cdata = '' )
    node = JTE_DOMCDATASection.new( cdata )
    node.ownerDocument = self
    return node
  end
  
  
  def createElement( tagName, nodeValue = nil )
    node = JTE_DOMElement.new( tagName )
    node.ownerDocument = self
    
    unless nodeValue.nil?
      node.appendChild( self.createTextNode( nodeValue ) )
    end
    
    return node
  end
  
  
  def saveXML
    xml = "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n"
    
    self._tree( xml, @documentElement, '' )
    
    return xml
  end

  
  def _tree( xml, node, tab )
    case node.nodeType
      when JTE_DOM::XML_ELEMENT_NODE
        tag = [ node.tagName ]
        
        for n in 0 ... node.attributes._item.length
          name = self.escapeXML( node.attributes._item[ n ].name )
          value = node.attributes._item[ n ].value
            
          tag.push( "#{name}=\"#{value}\"" )
        end
        
        if node.childNodes.length == 0
            xml.concat( "#{tab}<" + tag.join( ' ' ) + "/>\n" )
        else
          if ( node.childNodes.length == 1 ) && ( node.childNodes._item[ 0 ].nodeType != JTE_DOM::XML_ELEMENT_NODE )
            xml.concat( "#{tab}<" + tag.join( ' ' ) + ">" )
            self._tree( xml, node.childNodes._item[ 0 ], "" )
            xml.concat( "</#{node.tagName}>\n" )
          else
            xml.concat( "#{tab}<" + tag.join( ' ' ) + ">\n" )
            
            for n in 0 ... node.childNodes._item.length
              self._tree( xml, node.childNodes._item[ n ], "#{tab}  " )
            end
            
            xml.concat( "#{tab}</#{node.tagName}>\n" )
          end
        end
      
      when JTE_DOM::XML_COMMENT_NODE
        xml.concat( "<!--#{node.nodeValue}-->" )
      
      when JTE_DOM::XML_CDATA_SECTION_NODE
        xml.concat( "<![CDATA[#{node.nodeValue}]]>" )
      
      else
        xml.concat( self.escapeXML( node.nodeValue ).sub( /&#039;/, '&apos;' ) )
    end
  end
  
  
  def escapeXML( text )
    text.gsub( /&/, '&amp;' )
    text.gsub( /</, '&lt;' )
    text.gsub( />/, '&gt;' )
    text.gsub( /"/, '&quot;' )
    text.gsub( /'/, '&apos;' )
    
    return text
  end
  
  attr_accessor :docType, :encoding, :documentElement
end
